#include "signalsslots.h"

#include "../miscellaneous.h"
#include "math.h"

#include <QDebug>

EtherCATSignalsSlots::EtherCATSignalsSlots (VariablesInput& input, VariablesOutput& output, QObject *parent)
	: QObject(parent), input(input), output(output)
{
	
	connect(&input.fwdPowerSetPoint, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_power_watt(1, (double) value);
	});

	connect(&input.frequencySetPoint, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_frequency(1, (double) value);
	});
	
	connect(&input.sweepStartFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_start(1, (double) value);
	});
	
	connect(&input.sweepStopFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_stop(1, (double) value);
	});
	
	connect(&input.sweepStepFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_step(1, (double) value);
	});
	
	connect(&input.sweepPower, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_power(1, (double) value);
	});
	
	connect(&input.dllLowerFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_limit_lower(1, (double) value);
	});
	
	connect(&input.dllUpperFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_limit_upper(1, (double) value);
	});
	
	connect(&input.dllStartFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_start(1, (double) value);
	});
	
	connect(&input.dllStepFrequency, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_step(1, (double) value);
	});
	
	connect(&input.dllThreshold, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_threshold(1, (double) value);
	});
	
	connect(&input.dllDelay, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_DLL_delay(1, (double) value);
	});
	
	connect(&input.clockSource, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_clock_source(1, (int) value);
	});
	
	connect(&input.pwmDutyCycle, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_PWM_duty_cycle(1, (int) value);
	});
	
	connect(&input.phase, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_phase(1, (double) value);
	});
	
	connect(&input.phaseGainBoardPhase1, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_PGB_phase(1, 1, (double) value);
	});
	
	connect(&input.phaseGainBoardPhase2, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_PGB_phase(1, 2, (double) value);
	});
	
	connect(&input.phaseGainBoardPhase3, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_PGB_phase(1, 3, (double) value);
	});
	
	connect(&input.phaseGainBoardPhase4, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_PGB_phase(1, 4, (double) value);
	});
	
	connect(&input.commands, &EtherCATInputUInt16::changed, this, [this](quint16 value) //lambda...
	{
		for (int i=0; i<16; i++)
		{
			bool lastEnabled = lastCommands & (1 << i);
			bool enabled = value & (1 << i);

			if (enabled != lastEnabled)
			{
				switch (i)
				{
					// RF enable
					case 0:
						emit signal_set_RF_enable(1, enabled);
						break;

					// Reset
					case 1:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Reset SGx board";
							emit signal_execute_reset_SGx(1);
						}
						break;

					// Clear error
					case 2:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Clear Error";
							emit signal_execute_error_clear(1);
						}
						break;

					// Execute sweep
					case 3:
						if (enabled)
						{
							double startFreq = (double) this->input.sweepStartFrequency.getValue();
							double stopFreq = (double) this->input.sweepStopFrequency.getValue();
							double stepFreq = (double) this->input.sweepStepFrequency.getValue();
							double powWatt = (double) this->input.sweepPower.getValue();
							emit signal_execute_sweep(1, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;

					// DLL enable
					case 4:
						emit signal_set_DLL_enable(1, enabled);
						break;

					// PWM enable
					case 5:
						emit signal_set_PWM_enable(1, enabled);
						break;

					// No case 6 - placeholder for Software PID enable
//					case 6:
//						emit signal_set_PID_enable(1, enabled);
//						break;

					// Feed forward
					case 7:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
						}
						break;

					// Reset protection board
					case 8:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Reset Protection Board";
							emit signal_execute_reset_protection(1);
						}
						break;

					// PSU interlock
					case 9:
						emit signal_PSU_interlock(enabled);
						break;

					// Execute restart
					case 10:
						if (enabled)
						{
							emit signal_execute_restart_program();
						}
						break;

					//TX bit - Intended to be toggled as a heartbeat of sorts.
					case 15:
						this->output.status.setBit(15, enabled);
						break;
				}
			}
		}

		lastCommands = value;
	});	//end lambda
}

void EtherCATSignalsSlots::handler_generator_ready_get(bool enable)
{
	output.status.setBit(0, enable);
}

void EtherCATSignalsSlots::handler_RF_enable_get(int intrasys_num, bool enable)
{
	output.status.setBit(1, enable);
}

void EtherCATSignalsSlots::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	output.status.setBit(4, enable);
}

void EtherCATSignalsSlots::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	output.status.setBit(5, duty_cycle >= 1 && duty_cycle <= 99);
	output.pwmFrequency.setValue(frequency_hz);
	output.pwmDutyCycle.setValue(duty_cycle);
}

void EtherCATSignalsSlots::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	output.fwdPowerSetPoint.setValue(power_watt);
}

void EtherCATSignalsSlots::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
	output.fwdPower.setValue(PA_power_fwd_watt);
	output.rflPower.setValue(PA_power_rfl_watt);
	output.returnLoss.setValue(std::min(PA_s11_loss, 0.0));
}

void EtherCATSignalsSlots::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	output.actualFrequency.setValue(frequency_mhz);
}

void EtherCATSignalsSlots::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		QStringList SWP_data = SWP_raw_data.split("\r\n");

		QVector<double>	SWP_freq_data;
		QVector<double>	SWP_frw_data;
		QVector<double>	SWP_rfl_data;
		QVector<double>	SWP_s11_dbm_data;

		SWP_freq_data.resize(SWP_data.count()-2);
		SWP_frw_data.resize(SWP_data.count()-2);
		SWP_rfl_data.resize(SWP_data.count()-2);
		SWP_s11_dbm_data.resize(SWP_data.count()-2);

		QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(\\d+.?\\d+),(\\d+.?\\d+)");

		for (int i = 0; i < SWP_data.count() - 2; i++)
		{
			 if (regexp.indexIn(SWP_data.at(i))>=0)
			 {
				  QString string1 = regexp.cap(1);
				  QString string2 = regexp.cap(2);
				  QString string3 = regexp.cap(3);

				  SWP_freq_data[i] = string1.toDouble();
				  SWP_frw_data[i] = string2.toDouble();
				  SWP_rfl_data[i] = string3.toDouble();

				  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			 }
		}

		double frequency_best_match = 0;
		double s11_best_match = __INT_MAX__;

		for (int i = 0; i < SWP_s11_dbm_data.size(); i++)
		{
			if (s11_best_match > SWP_s11_dbm_data.at(i))
			{
				s11_best_match = SWP_s11_dbm_data.at(i);		//finds the best S11 match (lowest S11 in dB)
				frequency_best_match = SWP_freq_data.at(i);		//set frequency to frequency of best S11 match
			}
		}

		//S11 should be negative; Cap off S11 values to 0 Maximum;
		//Make the negative S11 value absolute (force positive), because registers only support positive values.
		if (s11_best_match > 0)
		{
			s11_best_match = 0;
		}

		double s11_start = SWP_s11_dbm_data.at(0);
		if (s11_start > 0)
		{
			s11_start = 0;
		}

		double s11_stop = SWP_s11_dbm_data.at(SWP_s11_dbm_data.size()-1);
		if (s11_stop > 0)
		{
			s11_stop = 0;
		}

		// Write Sweep data
		output.s11Frequency.setValue(frequency_best_match);
		output.s11Amplitude.setValue(s11_best_match);
		output.amplitudeSweepStartFrequency.setValue(s11_start);
		output.amplitudeSweepStopFrequency.setValue(s11_stop);
	}
}

void EtherCATSignalsSlots::handler_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	output.sweepStartFrequency.setValue(frequency_start);
	output.sweepStopFrequency.setValue(frequency_stop);
	output.sweepStepFrequency.setValue(frequency_step);
	output.sweepPower.setValue(power_watt);
}

void EtherCATSignalsSlots::handler_temperature_get(int intrasys_num, double temperature)
{
	output.temperature.setValue(round(temperature * 10.0));
}

void EtherCATSignalsSlots::handler_error_get(int intrasys_num, int error, QStringList error_messages)
{
	output.sgError.setValue(error);
}

void EtherCATSignalsSlots::handler_datalogging_enable_get(bool enable)
{
	output.status.setBit(7, enable);
}

void EtherCATSignalsSlots::handler_datalogging_storage_sufficient(bool enable)
{
	output.status.setBit(13, enable);
}

void EtherCATSignalsSlots::handler_PSU_enable_combined_get(int intrasys_num, bool enable)
{
	output.status.setBit(2, enable);
}

void EtherCATSignalsSlots::handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power)
{
	if (psu_num < 0 || psu_num > 3) return;
	output.psuVoltage[psu_num]->setValue(round(voltage * 10.0));
	output.psuCurrent[psu_num]->setValue(round(current * 10.0));
	output.psuPower[psu_num]->setValue(round(power));
}

void EtherCATSignalsSlots::handler_PSU_dissipation_get(int intrasys_num, double dissipation)
{
	output.psuPowerDissipation.setValue(std::max(round(dissipation), 0.0));
}

void EtherCATSignalsSlots::handler_PSU_power_efficiency_get(int intrasys_num, double efficiency)
{
	output.psuPowerEfficiency.setValue(round(efficiency * 10.0));
}

void EtherCATSignalsSlots::handler_SGx_communication_working_get(bool working)
{
	output.status.setBit(11, working);
}

void EtherCATSignalsSlots::handler_PSU_communication_working_get(bool working)
{
	output.status.setBit(12, working);
}

void EtherCATSignalsSlots::handler_get_phase(int intrasys_num, double phase_degrees)
{
	output.phase.setValue(round(phase_degrees));
}

void EtherCATSignalsSlots::handler_get_clock_source(int intrasys_num, int clock_source)
{
	output.clockSource.setValue(clock_source);
}

void EtherCATSignalsSlots::handler_get_PGB_phase(int intrasys_num, int pg_channel_num, double phase)
{
	if (pg_channel_num < 1 || pg_channel_num > 4) return;
	output.phaseGainBoardPhase[pg_channel_num-1]->setValue(((int) round(phase)) % 360);
}

/**********************************************************************************************************************************************************************************
 * 1Channel Variables
 * *******************************************************************************************************************************************************************************/
EtherCATSignalsSlots_1channel::EtherCATSignalsSlots_1channel (VariablesInput_1channel& input, VariablesOutput_1channel& output, QObject *parent)
	: QObject(parent), input(input), output(output)
{

	connect(&input.power_watt_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_power_watt(1, (double) value);
	});

	connect(&input.frequency_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_frequency(1, (double) value);
	});

	connect(&input.SWP_freq_lower_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_start(1, (double) value);
	});

	connect(&input.SWP_freq_upper_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_stop(1, (double) value);
	});

	connect(&input.SWP_freq_step_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_frequency_step(1, (double) value);
	});

	connect(&input.SWP_power_watt_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_SWP_power(1, (double) value);
	});

	connect(&input.DLL_freq_lower_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_limit_lower(1, (double) value);
	});

	connect(&input.DLL_freq_upper_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_limit_upper(1, (double) value);
	});

	connect(&input.DLL_freq_start_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_start(1, (double) value);
	});

	connect(&input.DLL_freq_step_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_frequency_step(1, (double) value);
	});

	connect(&input.DLL_threshold_set_CH1, &EtherCATInputFloat::changed, this, [this](float value) {
		emit signal_set_DLL_threshold(1, (double) value);
	});

	connect(&input.DLL_delay_set_CH1, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_DLL_delay(1, (double) value);
	});

	connect(&input.clock_source_set_CH1, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_clock_source(1, (int) value);
	});

	connect(&input.PWM_duty_cycle_set_CH1, &EtherCATInputUInt8::changed, this, [this](quint8 value) {
		emit signal_set_PWM_duty_cycle(1, (int) value);
	});

	connect(&input.phase_set_CH1, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
		emit signal_set_phase(1, (double) value);
	});

//	connect(&input.splitter_phase1_set, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_PGB_phase(1, 1, (double) value);
//	});

//	connect(&input.splitter_phase2_set, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_PGB_phase(1, 2, (double) value);
//	});

//	connect(&input.splitter_phase3_set, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_PGB_phase(1, 3, (double) value);
//	});

//	connect(&input.splitter_phase4_set, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_PGB_phase(1, 4, (double) value);
//	});

	connect(&input.commands_set_bitmask, &EtherCATInputUInt16::changed, this, [this](quint16 value) //lambda...
	{
		for (int i=0; i<16; i++)
		{
			bool lastEnabled = last_commands_set_bitmask & (1 << i);
			bool enabled = value & (1 << i);

			if (enabled != lastEnabled)
			{
				switch (i)
				{
					// RF enable
					case 0:
						emit signal_set_RF_enable(1, enabled);
						break;

					// Reset
					case 1:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Reset SGx board";
							emit signal_execute_reset_SGx(1);
						}
						break;

					// Clear error
					case 2:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Clear Error";
							emit signal_execute_error_clear(1);
						}
						break;

					// Execute sweep
					case 3:
						if (enabled)
						{
							double startFreq = (double) this->input.SWP_freq_lower_set_CH1.getValue();
							double stopFreq = (double) this->input.SWP_freq_upper_set_CH1.getValue();
							double stepFreq = (double) this->input.SWP_freq_step_set_CH1.getValue();
							double powWatt = (double) this->input.SWP_power_watt_set_CH1.getValue();
							emit signal_execute_sweep(1, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;

					// DLL enable
					case 4:
						emit signal_set_DLL_enable(1, enabled);
						break;

					// PWM enable
					case 5:
						emit signal_set_PWM_enable(1, enabled);
						break;

					// No case 6 - placeholder for Software PID enable
//					case 6:
//						emit signal_set_PID_enable(1, enabled);
//						break;

					// Feed forward
					case 7:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
						}
						break;

					// Reset protection board
					case 8:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Reset Protection Board";
							emit signal_execute_reset_protection(1);
						}
						break;

					// PSU interlock
					case 9:
						emit signal_PSU_interlock(enabled);
						break;

					// Execute restart
					case 10:
						if (enabled)
						{
							emit signal_execute_restart_program();
						}
						break;

					//TX bit - Intended to be toggled as a heartbeat of sorts.
					case 15:
						this->output.status_bitmask_CH1.setBit(15, enabled);
						break;
				}
			}
		}

		last_commands_set_bitmask = value;
	});	//end lambda
}

void EtherCATSignalsSlots_1channel::handler_generator_ready_get(bool enable)
{
	output.status_bitmask_CH1.setBit(0, enable);
}

void EtherCATSignalsSlots_1channel::handler_RF_enable_get(int intrasys_num, bool enable)
{
	output.status_bitmask_CH1.setBit(1, enable);
}

void EtherCATSignalsSlots_1channel::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	output.status_bitmask_CH1.setBit(4, enable);
}

void EtherCATSignalsSlots_1channel::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	output.status_bitmask_CH1.setBit(5, duty_cycle >= 1 && duty_cycle <= 99);
	output.PWM_frequency_get_CH1.setValue(frequency_hz);
	output.PWM_duty_cycle_get_CH1.setValue(duty_cycle);
}

void EtherCATSignalsSlots_1channel::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	output.power_watt_get_CH1.setValue(power_watt);
}

void EtherCATSignalsSlots_1channel::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
	output.FWD_power_watt_CH1.setValue(PA_power_fwd_watt);
	output.RFL_power_watt_CH1.setValue(PA_power_rfl_watt);
	output.S11_CH1.setValue(std::min(PA_s11_loss, 0.0));
}

void EtherCATSignalsSlots_1channel::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	output.frequency_get_CH1.setValue(frequency_mhz);
}

void EtherCATSignalsSlots_1channel::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		QStringList SWP_data = SWP_raw_data.split("\r\n");

		QVector<double>	SWP_freq_data;
		QVector<double>	SWP_frw_data;
		QVector<double>	SWP_rfl_data;
		QVector<double>	SWP_s11_dbm_data;

		SWP_freq_data.resize(SWP_data.count()-2);
		SWP_frw_data.resize(SWP_data.count()-2);
		SWP_rfl_data.resize(SWP_data.count()-2);
		SWP_s11_dbm_data.resize(SWP_data.count()-2);

		QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(\\d+.?\\d+),(\\d+.?\\d+)");

		for (int i = 0; i < SWP_data.count() - 2; i++)
		{
			 if (regexp.indexIn(SWP_data.at(i))>=0)
			 {
				  QString string1 = regexp.cap(1);
				  QString string2 = regexp.cap(2);
				  QString string3 = regexp.cap(3);

				  SWP_freq_data[i] = string1.toDouble();
				  SWP_frw_data[i] = string2.toDouble();
				  SWP_rfl_data[i] = string3.toDouble();

				  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			 }
		}

		double frequency_best_match = 0;
		double s11_best_match = __INT_MAX__;

		for (int i = 0; i < SWP_s11_dbm_data.size(); i++)
		{
			if (s11_best_match > SWP_s11_dbm_data.at(i))
			{
				s11_best_match = SWP_s11_dbm_data.at(i);		//finds the best S11 match (lowest S11 in dB)
				frequency_best_match = SWP_freq_data.at(i);		//set frequency to frequency of best S11 match
			}
		}

		//S11 should be negative; Cap off S11 values to 0 Maximum;
		//Make the negative S11 value absolute (force positive), because registers only support positive values.
		if (s11_best_match > 0)
		{
			s11_best_match = 0;
		}

		double s11_start = SWP_s11_dbm_data.at(0);
		if (s11_start > 0)
		{
			s11_start = 0;
		}

		double s11_stop = SWP_s11_dbm_data.at(SWP_s11_dbm_data.size()-1);
		if (s11_stop > 0)
		{
			s11_stop = 0;
		}

		// Write Sweep data
		output.SWP_frequency_best_match_CH1.setValue(frequency_best_match);
		output.SWP_S11_best_match_CH1.setValue(s11_best_match);
		output.SWP_S11_freq_lower_CH1.setValue(s11_start);
		output.SWP_S11_freq_upper_CH1.setValue(s11_stop);
	}
}

void EtherCATSignalsSlots_1channel::handler_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	output.SWP_freq_lower_get_CH1.setValue(frequency_start);
	output.SWP_freq_upper_get_CH1.setValue(frequency_stop);
	output.SWP_freq_step_get_CH1.setValue(frequency_step);
	output.SWP_power_watt_get_CH1.setValue(power_watt);
}

void EtherCATSignalsSlots_1channel::handler_temperature_get(int intrasys_num, double temperature)
{
	output.PA_temperature_CH1.setValue(round(temperature * 10.0));
}

void EtherCATSignalsSlots_1channel::handler_error_get(int intrasys_num, int error, QStringList error_messages)
{
	output.error_bitmask_CH1.setValue(error);
}

void EtherCATSignalsSlots_1channel::handler_datalogging_enable_get(bool enable)
{
	output.status_bitmask_CH1.setBit(7, enable);
}

void EtherCATSignalsSlots_1channel::handler_datalogging_storage_sufficient(bool enable)
{
	output.status_bitmask_CH1.setBit(13, enable);
}

void EtherCATSignalsSlots_1channel::handler_PSU_enable_combined_get(int intrasys_num, bool enable)
{
	output.status_bitmask_CH1.setBit(2, enable);
}

void EtherCATSignalsSlots_1channel::handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power)
{
	if (psu_num < 0 || psu_num > 3) return;
	output.psuVoltage[psu_num]->setValue(round(voltage * 10.0));
	output.psuCurrent[psu_num]->setValue(round(current * 10.0));
	output.psuPower[psu_num]->setValue(round(power));
}

void EtherCATSignalsSlots_1channel::handler_PSU_dissipation_get(int intrasys_num, double dissipation)
{
	output.PSU_power_dissipation.setValue(std::max(round(dissipation), 0.0));
}

void EtherCATSignalsSlots_1channel::handler_PSU_power_efficiency_get(int intrasys_num, double efficiency)
{
	output.PSU_power_efficiency.setValue(round(efficiency * 10.0));
}

void EtherCATSignalsSlots_1channel::handler_SGx_communication_working_get(bool working)
{
	output.status_bitmask_CH1.setBit(11, working);
}

void EtherCATSignalsSlots_1channel::handler_PSU_communication_working_get(bool working)
{
	output.status_bitmask_CH1.setBit(12, working);
}

void EtherCATSignalsSlots_1channel::handler_get_phase(int intrasys_num, double phase_degrees)
{
	output.phase_get_CH1.setValue(round(phase_degrees));
}

void EtherCATSignalsSlots_1channel::handler_get_clock_source(int intrasys_num, int clock_source)
{
	output.clock_source_get_CH1.setValue(clock_source);
}

//void EtherCATSignalsSlots_1channel::handler_get_PGB_phase(int intrasys_num, int pg_channel_num, double phase)
//{
//	if (pg_channel_num < 1 || pg_channel_num > 4) return;
//	output.phaseGainBoardPhase[pg_channel_num-1]->setValue(((int) round(phase)) % 360);
//}


/**********************************************************************************************************************************************************************************
 * 4Channel Variables
 * *******************************************************************************************************************************************************************************/

/* 4Channel EtherCAT Inputs / Sets */
EtherCATSignalsSlots_4channel::EtherCATSignalsSlots_4channel (VariablesInput_4channel& input, VariablesOutput_4channel& output, QObject *parent)
	: QObject(parent), input_4channel(input), output_4channel(output)
{
	connect(&input.frequency_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_frequency(1, value);
	});

	connect(&input.frequency_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_frequency(2, value);
	});

	connect(&input.frequency_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_frequency(3, value);
	});

	connect(&input.frequency_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_frequency(4, value);
	});



	connect(&input.power_watt_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_power_watt(1, value);
	});

	connect(&input.power_watt_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_power_watt(2, value);
	});

	connect(&input.power_watt_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_power_watt(3, value);
	});

	connect(&input.power_watt_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_power_watt(4, value);
	});



//	connect(&input.phase_set_CH1, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_phase(1, (double) value);
//	});

//	connect(&input.phase_set_CH2, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_phase(2, (double) value);
//	});

//	connect(&input.phase_set_CH3, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_phase(3, (double) value);
//	});

//	connect(&input.phase_set_CH4, &EtherCATInputUInt16::changed, this, [this](quint16 value) {
//		emit signal_set_phase(4, (double) value);
//	});



	connect(&input.VGA_attenuation_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_VGA_attenuation(1, value);
	});

	connect(&input.VGA_attenuation_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_VGA_attenuation(2, value);
	});

	connect(&input.VGA_attenuation_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_VGA_attenuation(3, value);
	});

	connect(&input.VGA_attenuation_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_VGA_attenuation(4, value);
	});



	connect(&input.IQMod_magnitude_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_IQMod_magnitude(1, value);
	});

	connect(&input.IQMod_magnitude_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_IQMod_magnitude(2, value);
	});

	connect(&input.IQMod_magnitude_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_IQMod_magnitude(3, value);
	});

	connect(&input.IQMod_magnitude_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value)
	{
		value = value * 0.1;
		emit signal_set_IQMod_magnitude(4, value);
	});



	connect(&input.DLL_freq_lower_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_lower(1, value);
	});

	connect(&input.DLL_freq_lower_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_lower(2, value);
	});

	connect(&input.DLL_freq_lower_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_lower(3, value);
	});

	connect(&input.DLL_freq_lower_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_lower(4, value);
	});



	connect(&input.DLL_freq_upper_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_upper(1, value);
	});

	connect(&input.DLL_freq_upper_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_upper(2, value);
	});

	connect(&input.DLL_freq_upper_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_upper(3, value);
	});

	connect(&input.DLL_freq_upper_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_limit_upper(4, value);
	});



	connect(&input.DLL_freq_start_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_start(1, value);
	});

	connect(&input.DLL_freq_start_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_start(2, value);
	});

	connect(&input.DLL_freq_start_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_start(3, value);
	});

	connect(&input.DLL_freq_start_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_start(4, value);
	});



	connect(&input.DLL_freq_step_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_step(1, value);
	});

	connect(&input.DLL_freq_step_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_step(2, value);
	});

	connect(&input.DLL_freq_step_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_step(3, value);
	});

	connect(&input.DLL_freq_step_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_frequency_step(4, value);
	});



	connect(&input.DLL_threshold_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_threshold(1, value);
	});

	connect(&input.DLL_threshold_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_threshold(2, value);
	});

	connect(&input.DLL_threshold_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_threshold(3, value);
	});

	connect(&input.DLL_threshold_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_DLL_threshold(4, value);
	});



//	connect(&input.DLL_delay_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
//		emit signal_set_DLL_delay(1, value);
//	});

//	connect(&input.DLL_delay_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
//		emit signal_set_DLL_delay(2, value);
//	});

//	connect(&input.DLL_delay_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
//		emit signal_set_DLL_delay(3, value);
//	});

//	connect(&input.DLL_delay_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
//		emit signal_set_DLL_delay(4, value);
//	});


	connect(&input.SWP_freq_lower_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_start(1, value);
	});

	connect(&input.SWP_freq_lower_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_start(2, value);
	});

	connect(&input.SWP_freq_lower_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_start(3, value);
	});

	connect(&input.SWP_freq_lower_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_start(4, value);
	});



	connect(&input.SWP_freq_upper_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_stop(1, value);
	});

	connect(&input.SWP_freq_upper_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_stop(2, value);
	});

	connect(&input.SWP_freq_upper_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_stop(3, value);
	});

	connect(&input.SWP_freq_upper_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_stop(4, value);
	});



	connect(&input.SWP_freq_step_set_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_step(1, value);
	});

	connect(&input.SWP_freq_step_set_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_step(2, value);
	});

	connect(&input.SWP_freq_step_set_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_step(3, value);
	});

	connect(&input.SWP_freq_step_set_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_frequency_step(4, value);
	});



	connect(&input.SWP_power_watt_CH1, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_power(1, value);
	});

	connect(&input.SWP_power_watt_CH2, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_power(2, value);
	});

	connect(&input.SWP_power_watt_CH3, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_power(3, value);
	});

	connect(&input.SWP_power_watt_CH4, &EtherCATInputUInt16::changed, this, [this](double value) {
		value = value * 0.1;
		emit signal_set_SWP_power(4, value);
	});


	connect(&input.ERRC_SWP_execute_bitmask, &EtherCATInputUInt8::changed, this, [this](quint8 value)
	{
		for (int i=0; i<8; i++)
		{
			bool lastEnabled = last_ERRC_SWP_execute_bitmask & (1 << i);
			bool enabled = value & (1 << i);
			if (enabled != lastEnabled)
			{
				switch (i)
				{
					case 0:
						if (enabled){
							emit signal_execute_error_clear(1);
						}
						break;
					case 1:
						if (enabled){
							emit signal_execute_error_clear(2);
						}
						break;
					case 2:
						if (enabled){
							emit signal_execute_error_clear(3);
						}
						break;
					case 3:
						if (enabled){
							emit signal_execute_error_clear(4);
						}
						break;
					case 4:
						if (enabled)
						{
							double startFreq = 0.1 * (double)this->input_4channel.SWP_freq_lower_set_CH1.getValue();
							double stopFreq = 0.1 * (double)this->input_4channel.SWP_freq_upper_set_CH1.getValue();
							double stepFreq = 0.1 * (double)this->input_4channel.SWP_freq_step_set_CH1.getValue();
							double powWatt = 0.1 * (double)this->input_4channel.SWP_power_watt_CH1.getValue();
							emit signal_execute_sweep(1, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;
					case 5:
						if (enabled)
						{
							double startFreq = 0.1 * (double)this->input_4channel.SWP_freq_lower_set_CH2.getValue();
							double stopFreq = 0.1 * (double)this->input_4channel.SWP_freq_upper_set_CH2.getValue();
							double stepFreq = 0.1 * (double)this->input_4channel.SWP_freq_step_set_CH2.getValue();
							double powWatt = 0.1 * (double)this->input_4channel.SWP_power_watt_CH2.getValue();
							emit signal_execute_sweep(2, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;
					case 6:
						if (enabled)
						{
							double startFreq = 0.1 * (double)this->input_4channel.SWP_freq_lower_set_CH3.getValue();
							double stopFreq = 0.1 * (double)this->input_4channel.SWP_freq_upper_set_CH3.getValue();
							double stepFreq = 0.1 * (double)this->input_4channel.SWP_freq_step_set_CH3.getValue();
							double powWatt = 0.1 * (double)this->input_4channel.SWP_power_watt_CH3.getValue();
							emit signal_execute_sweep(3, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;
					case 7:
						if (enabled)
						{
							double startFreq = 0.1 * (double)this->input_4channel.SWP_freq_lower_set_CH4.getValue();
							double stopFreq = 0.1 * (double)this->input_4channel.SWP_freq_upper_set_CH4.getValue();
							double stepFreq = 0.1 * (double)this->input_4channel.SWP_freq_step_set_CH4.getValue();
							double powWatt = 0.1 * (double)this->input_4channel.SWP_power_watt_CH4.getValue();
							emit signal_execute_sweep(4, startFreq, stopFreq, stepFreq, convert_watt_to_dbm(powWatt));
						}
						break;
					default:
						break;
				}
			}
		}

		last_ERRC_SWP_execute_bitmask = value;

	});		//end Lambda


	connect(&input.RFE_DLL_set_bitmask, &EtherCATInputUInt8::changed, this, [this](quint8 value)
	{
		for (int i=0; i<8; i++)
		{
			bool lastEnabled = last_RFE_DLL_set_bitmask & (1 << i);
			bool enabled = value & (1 << i);

			if (enabled != lastEnabled)
			{
				switch (i)
				{
					case 0:
						emit signal_set_RF_enable(1, enabled);
						break;
					case 1:
						emit signal_set_RF_enable(2, enabled);
						break;
					case 2:
						emit signal_set_RF_enable(3, enabled);
						break;
					case 3:
						emit signal_set_RF_enable(4, enabled);
						break;
					case 4:
						emit signal_set_DLL_enable(1, enabled);
						break;
					case 5:
						emit signal_set_DLL_enable(2, enabled);
						break;
					case 6:
						emit signal_set_DLL_enable(3, enabled);
						break;
					case 7:
						emit signal_set_DLL_enable(4, enabled);
						break;
					default:
						break;
				}
			}
		}

		last_RFE_DLL_set_bitmask = value;

	});		//end Lambda


	connect(&input.feedforward_enable_set_bitmask, &EtherCATInputUInt8::changed, this, [this](quint8 value)
	{
		for (int i=0; i<8; i++)
		{
			bool lastEnabled = last_feedforward_enable_set_bitmask & (1 << i);
			bool enabled = value & (1 << i);

			if (enabled != lastEnabled)
			{
				switch (i)
				{
					case 0:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
						}
						break;
					case 1:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(2, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(2, POWER_CONTROL_NORMAL);
						}
						break;
					case 2:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(3, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(3, POWER_CONTROL_NORMAL);
						}
						break;
					case 3:
						if (enabled)
						{
							qDebug() << "EtherCAT -> Power Control Mode: Feed Forward";
							emit signal_set_power_control_mode(4, POWER_CONTROL_FEED_FWD);
						}
						else
						{
							qDebug() << "EtherCAT -> Power Control Mode: Autogain";
							emit signal_set_power_control_mode(4, POWER_CONTROL_NORMAL);
						}						break;
					default:
						break;
				}
			}
		}

		last_feedforward_enable_set_bitmask = value;

	});		//end Lambda


}	//end of EtherCATSignalsSlots_4channel constructor


/* 4Channel EtherCAT outputs / reads */
void EtherCATSignalsSlots_4channel::handler_power_control_mode_get(int intrasys_num, int mode)
{
	int channel_bit = intrasys_num - 1;
	if (channel_bit < 4)
	{
		if (mode == POWER_CONTROL_NORMAL)
		{
			output_4channel.feedforward_enable_get_bitmask.setBit(channel_bit, 0);
		}
		else
		{
			output_4channel.feedforward_enable_get_bitmask.setBit(channel_bit, 1);
		}
	}
}


/* 4Channel EtherCAT outputs / reads */
void EtherCATSignalsSlots_4channel::handler_RF_enable_get(int intrasys_num, bool enable)
{
	int channel_bit = intrasys_num - 1;
	if (channel_bit < 4)
	{
		output_4channel.RFE_DLL_get_bitmask.setBit(channel_bit, enable);
	}
}

void EtherCATSignalsSlots_4channel::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	int channel_bit = intrasys_num - 1 + 4;
	if (channel_bit < 8)
	{
		output_4channel.RFE_DLL_get_bitmask.setBit(channel_bit, enable);
	}
}


void EtherCATSignalsSlots_4channel::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	switch (intrasys_num)
	{
		case 1:
			output_4channel.error_status_CH1.setValue(error);
			break;
		case 2:
			output_4channel.error_status_CH2.setValue(error);
			break;
		case 3:
			output_4channel.error_status_CH3.setValue(error);
			break;
		case 4:
			output_4channel.error_status_CH4.setValue(error);
			break;
		default:
			break;
	}
}

void EtherCATSignalsSlots_4channel::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	quint16 frequency = (quint16) round(frequency_mhz * 10);
	switch (intrasys_num)
	{
		case 1:
			output_4channel.frequency_get_CH1.setValue(frequency);
			break;
		case 2:
			output_4channel.frequency_get_CH2.setValue(frequency);
			break;
		case 3:
			output_4channel.frequency_get_CH3.setValue(frequency);
			break;
		case 4:
			output_4channel.frequency_get_CH4.setValue(frequency);
			break;
		default:
			break;
	}
}

void EtherCATSignalsSlots_4channel::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	quint16 power = (quint16) round(power_watt * 10);		//Handle the rounding properly
	switch (intrasys_num)
	{
		case 1:
			output_4channel.power_watt_get_CH1.setValue(power);
			break;
		case 2:
			output_4channel.power_watt_get_CH2.setValue(power);
			break;
		case 3:
			output_4channel.power_watt_get_CH3.setValue(power);
			break;
		case 4:
			output_4channel.power_watt_get_CH4.setValue(power);
			break;
		default:
			break;
	}
}

void EtherCATSignalsSlots_4channel::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
	quint16 power_FWD_watt = (quint16) round(PA_power_fwd_watt * 10);		//Handle the rounding properly
	quint16 power_RFL_watt = (quint16) round(PA_power_rfl_watt * 10);		//Handle the rounding properly

	switch (intrasys_num)
	{
		case 1:
			output_4channel.FWD_power_watt_CH1.setValue(power_FWD_watt);
			output_4channel.RFL_power_watt_CH1.setValue(power_RFL_watt);
			break;
		case 2:
			output_4channel.FWD_power_watt_CH2.setValue(power_FWD_watt);
			output_4channel.RFL_power_watt_CH2.setValue(power_RFL_watt);
			break;
		case 3:
			output_4channel.FWD_power_watt_CH3.setValue(power_FWD_watt);
			output_4channel.RFL_power_watt_CH3.setValue(power_RFL_watt);
			break;
		case 4:
			output_4channel.FWD_power_watt_CH4.setValue(power_FWD_watt);
			output_4channel.RFL_power_watt_CH4.setValue(power_RFL_watt);
			break;
		default:
			break;
	}
}

//void EtherCATSignalsSlots_4channel::handler_get_phase(int intrasys_num, double phase_degrees)
//{
//	switch (intrasys_num)
//	{
//		case 1:
//			output_4channel.phase_get_CH1.setValue(round(phase_degrees));
//			break;
//		case 2:
//			output_4channel.phase_get_CH2.setValue(round(phase_degrees));
//			break;
//		case 3:
//			output_4channel.phase_get_CH3.setValue(round(phase_degrees));
//			break;
//		case 4:
//			output_4channel.phase_get_CH4.setValue(round(phase_degrees));
//			break;
//		default:
//			break;
//	}
//}


void EtherCATSignalsSlots_4channel::handler_VGA_attenuation_get(int intrasys_num, double attenuation)
{
	quint16 attenuation_value = (quint16) round(attenuation * 10);		//Handle the rounding properly
	switch (intrasys_num)
	{
		case 1:
			output_4channel.VGA_attenuation_get_CH1.setValue(attenuation_value);
			break;
		case 2:
			output_4channel.VGA_attenuation_get_CH2.setValue(attenuation_value);
			break;
		case 3:
			output_4channel.VGA_attenuation_get_CH3.setValue(attenuation_value);
			break;
		case 4:
			output_4channel.VGA_attenuation_get_CH4.setValue(attenuation_value);
			break;
		default:
			break;
	}
}

void EtherCATSignalsSlots_4channel::handler_IQMod_magnitude_get(int intrasys_num, double magnitude)
{
	quint16 magnitude_value = (quint16) round(magnitude * 10);		//Handle the rounding properly
	switch (intrasys_num)
	{
		case 1:
			output_4channel.IQMod_magnitude_get_CH1.setValue(magnitude_value);
			break;
		case 2:
			output_4channel.IQMod_magnitude_get_CH2.setValue(magnitude_value);
			break;
		case 3:
			output_4channel.IQMod_magnitude_get_CH3.setValue(magnitude_value);
			break;
		case 4:
			output_4channel.IQMod_magnitude_get_CH4.setValue(magnitude_value);
			break;
		default:
			break;
	}
}

void EtherCATSignalsSlots_4channel::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		QStringList SWP_data = SWP_raw_data.split("\r\n");

		QVector<double>	SWP_freq_data;
		QVector<double>	SWP_frw_data;
		QVector<double>	SWP_rfl_data;
		QVector<double>	SWP_s11_dbm_data;

		SWP_freq_data.resize(SWP_data.count()-2);
		SWP_frw_data.resize(SWP_data.count()-2);
		SWP_rfl_data.resize(SWP_data.count()-2);
		SWP_s11_dbm_data.resize(SWP_data.count()-2);

		QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(\\d+.?\\d+),(\\d+.?\\d+)");

		for (int i = 0; i < SWP_data.count() - 2; i++)
		{
			 if (regexp.indexIn(SWP_data.at(i))>=0)
			 {
				  QString string1 = regexp.cap(1);
				  QString string2 = regexp.cap(2);
				  QString string3 = regexp.cap(3);

				  SWP_freq_data[i] = string1.toDouble();
				  SWP_frw_data[i] = string2.toDouble();
				  SWP_rfl_data[i] = string3.toDouble();

				  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			 }
		}

		double frequency_best_match = 0;
		double s11_best_match = __INT_MAX__;

		for (int i = 0; i < SWP_s11_dbm_data.size(); i++)
		{
			if (s11_best_match > SWP_s11_dbm_data.at(i))
			{
				s11_best_match = SWP_s11_dbm_data.at(i);		//finds the best S11 match (lowest S11 in dB)
				frequency_best_match = SWP_freq_data.at(i);		//set frequency to frequency of best S11 match
			}
		}
		frequency_best_match = round(frequency_best_match * 10);


		//S11 should be negative; Cap off S11 values to 0 Maximum;
		//Make the negative S11 value absolute (force positive), because registers only support positive values.
		if (s11_best_match > 0)
		{
			s11_best_match = 0;
		}
		s11_best_match = round(abs(s11_best_match) * 10);


		double s11_start = SWP_s11_dbm_data.at(0);
		if (s11_start > 0)
		{
			s11_start = 0;
		}
//		s11_start = round((abs(s11_start) * multiplier_map.value("IR_S11_loss_start_freq")));


		double s11_stop = SWP_s11_dbm_data.at(SWP_s11_dbm_data.size()-1);
		if (s11_stop > 0)
		{
			s11_stop = 0;
		}
//		s11_stop = round((abs(s11_stop) * multiplier_map.value("IR_S11_loss_stop_freq")));


		// Write Sweep data
		switch (intrasys_num)
		{
			case 1:
				output_4channel.Best_S11_Freq_CH1.setValue((quint16)frequency_best_match);
				output_4channel.Best_S11_dB_CH1.setValue((quint16)s11_best_match);
//				output_4channel.amplitudeSweepStartFrequency.setValue(s11_start);
//				output_4channel.amplitudeSweepStopFrequency.setValue(s11_stop);
				break;
			case 2:
				output_4channel.Best_S11_Freq_CH2.setValue((quint16)frequency_best_match);
				output_4channel.Best_S11_dB_CH2.setValue((quint16)s11_best_match);
//				output_4channel.amplitudeSweepStartFrequency.setValue(s11_start);
//				output_4channel.amplitudeSweepStopFrequency.setValue(s11_stop);
				break;
			case 3:
				output_4channel.Best_S11_Freq_CH3.setValue((quint16)frequency_best_match);
				output_4channel.Best_S11_dB_CH3.setValue((quint16)s11_best_match);
//				output_4channel.amplitudeSweepStartFrequency.setValue(s11_start);
//				output_4channel.amplitudeSweepStopFrequency.setValue(s11_stop);
				break;
			case 4:
				output_4channel.Best_S11_Freq_CH4.setValue((quint16)frequency_best_match);
				output_4channel.Best_S11_dB_CH4.setValue((quint16)s11_best_match);
//				output_4channel.amplitudeSweepStartFrequency.setValue(s11_start);
//				output_4channel.amplitudeSweepStopFrequency.setValue(s11_stop);
				break;
			default:
				break;
		}
	}
}

